#!/usr/bin/env runhaskell

import AssertGen.Parser
import AssertGen.AST
import AssertGen.SimpleAssertGen
import Control.Monad
import Data.Char (isUpper)
import Data.Maybe (fromJust)
import qualified Data.Text as T
import qualified Data.Text.Encoding as E
import qualified Data.ByteString as BS
import Data.List (find, (\\), nub, stripPrefix)
import System.IO (stdin)
import System.Environment
import System.Exit

extractSig :: String -> [Decl] -> Either String (Fam, [Inhabitant])
extractSig famName ds = do
    fterm <- case find (isFamDecl famName) ds of
               Nothing -> Left $ "No family with name '" ++ famName ++ "'"
               Just (DDecl _ fterm) -> return fterm
    famArgs <- extFamArgs fterm
    let fam = Fam famName famArgs
    let famInhs = foldr (extInh fam) [] ds
    return (fam, famInhs)
    where
      isFamDecl name (DDecl name' _) = name == name'
      isFamDecl _ _ = False

      extFamArgs TType = Right []
      extFamArgs (TArrow (TConst n) t') = do
          as <- extFamArgs t'
          return (n:as)
      extFamArgs _ = Left "Unsupported format for type family declaration"

      extInh fam (DDecl inhName inhTerm) is
          | TConst famName' <- root (conc inhTerm), famName' == famName
              = (Inh fam (fromJust $ stripPrefix (famName++"/") inhName)
                         (nub $ freeMetas [] inhTerm)
                         (prems inhTerm)
                         (argList (conc inhTerm))):is
          | otherwise
              = is
      extInh _ _ is = is

      prems (TArrow t1 t2) = t1:prems t2
      prems _ = []

      freeMetas as (TPi ident _ t2) = freeMetas (ident:as) t2
      freeMetas as (TArrow t1 t2) = freeMetas as t1 ++ freeMetas as t2
      freeMetas as (TApp t1 t2) = freeMetas as t1 ++ freeMetas as t2
      freeMetas _ (TConst _) = []
      freeMetas as (TVar v) = if isUpper (head v) && not (v `elem` as) then [v] else []
      freeMetas as (TLam ident _ t2) = freeMetas (ident:as) t2
      freeMetas _ THole = []
      freeMetas _ TType = []

main :: IO ()
main = do
  args <- getArgs
  when (not ("--" `elem` args) || length (tail $ dropWhile (/= "--") args) /= 7) $ do
         progName <- getProgName
         putStrLn $ "Usage: " ++ progName ++ " [FAMILIES] -- <DATA> <ASSERT> <ADMIT> <CUTELIM> <NORMDECL> <NORMCASES> <ABBREV>"
         putStrLn $ ""
         mapM_ (\x -> putStrLn $ "    " ++ x) $
               ["FAMILIES  :  Names of type families for which to generate boilerplate."
               ,"DATA      :  Output file for representation logic declarations."
               ,"ASSERT    :  Output file for assertion logic declarations."
               ,"ADMIT     :  Output file for cut admissibility cases."
               ,"CUTELIM   :  Output file for cut elimination cases."
               ,"NORMDECL  :  Output file for lemma declarations for repr. logic soundness theorems."
               ,"NORMCASES :  Output file for repr. logic soundness proof cases."
               ,"ABBREV    :  Output file for convenience proof abbreviations."]
         exitFailure
  let fams = takeWhile (/="--") args
  let [fData, fAssert, fAdmit, fCutElim, fNormDecl, fNormCases, fAbbrev] = tail $ dropWhile (/="--") args
  bs <- E.decodeUtf8 `liftM` BS.hGetContents stdin
  case parseSig initParserState "<stdin>" bs of
    Left err -> do
         putStrLn $ show err
         exitFailure
    Right (ds, _) -> do
         case mapM (flip extractSig ds) fams of
           Left err -> putStrLn (show err) >> exitFailure
           Right sigs -> do
               make fData fAssert fAdmit fCutElim fNormDecl fNormCases fAbbrev sigs
  return ()